﻿#!/usr/bin/python
# -*- coding: utf-8 -*-

""" NEW SONG ANNOUNCER

Announces in chat the new song being played with optional overlay integration.

Version
	1.4.1
		Making use of Parent.GetStreamingService() in version 1.0.2.30
	1.4.0
		Made compatible for version 1.0.2.29 and support Youtube Service
	1.3.2
		Fixed an issue with ascii encoding for some song titles
	1.3.1
		Adjusted script for the change of AnkhBot to Streamlabs Chatbot
	1.3.0
		Added $prevsong and $prevrequestedby parameter to be used in regular commands
		to be replaced with the previous song title and requester.
	1.2.1
		Updated reading settings file as utf-8-sig due to change also making use
		of the new ScriptToggled to prevent a message on enable
	1.2.0
		Added Spotify mode when using only Spotify Client to play music
	1.1.1
		Using os.getcwd() in init to fix textfile path locations
	1.1.0
		Added seperate response for if caster is the requester
	1.0.3
		Clean up code and added overlay example
	1.0.2
		Reading textfiles as unicode as song titles can contain unicode
	1.0.1
		Trimming whitespaces around song title and requester name
	1.0.0
		Initial public release

"""

#---------------------------------------
#	Import Libraries
#---------------------------------------
import clr
clr.AddReference("IronPython.Modules.dll")

import json
import os
import time
import codecs

#---------------------------------------
#	Script Information
#---------------------------------------
ScriptName = "New Song Announcer"
Website = "http://www.twitch.tv/ocgineer"
Creator = "Ocgineer"
Version = "1.4.1.0"
Description = "Announces in chat the new song being played with optional overlay integration."

#---------------------------------------
#	Script Global Variables
#---------------------------------------
SettingsFile = os.path.join(os.path.dirname(__file__), "NSASettings.json")
ReadMeFile = os.path.join(os.path.dirname(__file__), "ReadMe.txt")
ScriptSettings = None
TextFiles = None
SongInfo = None
TimeStamp = None
PreviousSong = None
CheckUnlocked = False

#---------------------------------------
#	Script Classes
#---------------------------------------
class Settings(object):
	""" Class to hold the script settings, matching UI_Config.json. """

	def __init__(self, settingsfile=None):
		""" Load in saved settings file if available else set default values. """
		try:
			with codecs.open(settingsfile, encoding="utf-8-sig", mode="r") as f:
				self.__dict__ = json.load(f, encoding="utf-8")
		except:
			self.MessageViewerRequester = "Now playing: {0} - Requested by {1}"
			self.MessageCasterRequester = "Now playing: {0}"
			self.FileCheck = 5
			self.FileMode = "ChatBot"

	def Reload(self, jsonData):
		""" Reload settings from the user interface by given json data. """
		self.__dict__ = json.loads(jsonData, encoding="utf-8")

	def ValidateValues(self):
		""" Validate set values in the current settings object class. """
		if self.FileCheck < 1:
			self.FileCheck = 1

class Info(object):
	""" Holds data of the current song being played. """
	def __init__(self):
		self.Title = ""
		self.Requester = ""
		self.IsPlaylist = False

	@classmethod
	def CopyInfo(cls, songInfo):
		""" Create a copy of songInfo, not a reference copy. """
		cls = Info()
		cls.Title = songInfo.Title
		cls.Requester = songInfo.Requester
		cls.IsPlaylist = songInfo.IsPlaylist
		return cls

class FilePaths(object):
	""" Holds the path to the various required textfiles. """
	def __init__(self, title, requester, spotify):
		self.SongTitle = title
		self.SongRequester = requester
		self.SongSpotify = spotify

#---------------------------------------
#	Script Functions
#---------------------------------------
def GetTextFileContent(textfile):
	""" Get the text content from the given textfile path. """
	try:
		with codecs.open(textfile, encoding="utf-8-sig", mode="r") as f:
			return f.readline().strip()
	except:
		return ""

def SetCurrentSong():
	""" Set the current song info from the textfiles for init. """
	global PreviousSong
	PreviousSong = None

	if ScriptSettings.FileMode == "ChatBot":
		SongInfo.Title = GetTextFileContent(TextFiles.SongTitle)
		SongInfo.Requester = GetTextFileContent(TextFiles.SongRequester)
	else:
		SongInfo.Title = GetTextFileContent(TextFiles.SongSpotify)
		SongInfo.Requester = Parent.GetChannelName()

	return

#---------------------------------------
#	Chatbot Initialize Function
#---------------------------------------
def Init():
	""" Initialize script or startup or reload. """

	# Globals
	global ScriptSettings
	global TextFiles
	global SongInfo
	global TimeStamp
	global CheckUnlocked

	# Lock checking files
	CheckUnlocked = False

	# Load saved settings and validate values
	ScriptSettings = Settings(SettingsFile)
	ScriptSettings.ValidateValues()

	# Get chatbot.exe dir path by using os.getcwd() but only in
	# Init() it will return the correct path.
	basedir = os.getcwd()

	# Set paths
	TextFiles = FilePaths(
		os.path.join(basedir, "Services\\{0}\\Files\\CurrentSong.txt".format(Parent.GetStreamingService())),
		os.path.join(basedir, "Services\\{0}\\Files\\RequestedBy.txt".format(Parent.GetStreamingService())),
		os.path.join(basedir, "Services\\{0}\\Files\\SpotifySong.txt".format(Parent.GetStreamingService()))
	)

	# Get 'current' song info
	SongInfo = Info()
	SetCurrentSong()

	# Set timestamp and unlock checking files
	TimeStamp = time.time()
	CheckUnlocked = True

	# End of Init
	return

#---------------------------------------
# Chatbot Save Settings Function
#---------------------------------------
def ReloadSettings(jsondata):
	""" Set newly saved data from UI after user saved settings. """

	# Globals
	global TimeStamp
	global CheckUnlocked

	# Lock checking files
	CheckUnlocked = False

	# Reload saved settings and validate values
	ScriptSettings.Reload(jsondata)
	ScriptSettings.ValidateValues()

	# Get 'current' song info
	SetCurrentSong()

	# Set timestamp and unlock checking files
	TimeStamp = time.time()
	CheckUnlocked = True

	# End of ReloadSettings
	return

#---------------------------------------
#	Chatbot Toggle Function
#---------------------------------------
def ScriptToggled(state):
	""" Handle enabling or disabling of the script state. """

	# Globals
	global TimeStamp
	global CheckUnlocked

	# Script Enabled
	if state:

		# Get 'current' song info
		SetCurrentSong()

		# Set timestamp and unlock checking files
		TimeStamp = time.time()
		CheckUnlocked = True

	# Script Disabled
	else:

		# Lock checking files
		CheckUnlocked = False

	# End of ScriptToggled
	return

#---------------------------------------
#	Chatbot Execute Function
#---------------------------------------
def Execute(data):
	""" Nothing to  do here. """
	return

#---------------------------------------
#	Chatbot Tick Function
#---------------------------------------
def Tick():
	""" Check textfile, determine new song, output to chat and websocket. """

	# Globals
	global TimeStamp
	global PreviousSong

	# Only check if script is enable and the
	# FileCheck seconds has elapsed to ease reading
	if CheckUnlocked and time.time() - TimeStamp >= ScriptSettings.FileCheck:

		# Set new timestamp
		TimeStamp = time.time()

		# Get current song title
		if ScriptSettings.FileMode == "ChatBot":
			currentSong = GetTextFileContent(TextFiles.SongTitle)
		else:
			currentSong = GetTextFileContent(TextFiles.SongSpotify)

		# Continue if it is a new song
		if currentSong != SongInfo.Title:

			# Set (copy) previous song
			PreviousSong = Info.CopyInfo(SongInfo)

			# Set the new song and get requester
			# If Spotify mode, set requester equal to channel
			SongInfo.Title = currentSong
			if ScriptSettings.FileMode == "ChatBot":
				SongInfo.Requester = GetTextFileContent(TextFiles.SongRequester)
			else:
				SongInfo.Requester = Parent.GetChannelName()

			# Requester is equal to the channel name the bot is connected to
			if Parent.GetChannelName() == SongInfo.Requester:

				# Post message in Twitch Chat and set boolean is playlist (true)
				Parent.SendStreamMessage(ScriptSettings.MessageCasterRequester.format(SongInfo.Title, SongInfo.Requester))
				SongInfo.IsPlaylist = True

			# Requester is a viewer
			else:

				# Post message in Twitch Chat and set boolean is request (false)
				Parent.SendStreamMessage(ScriptSettings.MessageViewerRequester.format(SongInfo.Title, SongInfo.Requester))
				SongInfo.IsPlaylist = False

			# Post new song data on websocket
			Parent.BroadcastWsEvent("EVENT_NOW_PLAYING", json.dumps(SongInfo.__dict__, encoding='utf-8', ensure_ascii=False))

	# End of Tick
	return

#---------------------------------------
# Chatbot Parse Function
#---------------------------------------
def Parse(parseString, user, target, message):
	""" This scripts custom command parameter parser. """

	# $previoussong - previous played song
	if "$prevsong" in parseString:

		# Replace parameter with resplacement string
		if PreviousSong:
			return parseString.replace("$prevsong", PreviousSong.Title)
		else:
			return parseString.replace("$prevsong", "none")

	# $previousrequestedby - requester of previous song
	if "$prevrequestedby" in parseString:

		# Replace parameter with resplacement string
		if PreviousSong:
			return parseString.replace("$prevrequestedby", PreviousSong.Requester)
		else:
			return parseString.replace("$prevrequestedby", "nobody")

	# Return unaltered parseString
	return parseString

#---------------------------------------
# Script UI Button Functions
#---------------------------------------
def OpenReadMe():
	""" Open the script readme file in users default .txt application. """

	os.startfile(ReadMeFile)
	return
