#!/usr/bin/python
# -*- coding: utf-8 -*-

""" TWITCH SUBSCRIPTION TRACKER

Version:
	1.0.3
		Fixed bug in $tstTopGifters displaying subbed months instead the number of
		gifted subs. Fixed readme button and updated readme itself.

	1.0.2
		Adjusted chat output for chatbot version 1.0.2.29

	1.0.1
		Bugfix on updating existing user in the database.

	1.0.0
		Initial public release.
		This script is based on the Total Sub Month Tracker but due to a
		rewrite it is incompatable. Mainly due to tracking of gifted subs.

"""

#---------------------------------------
# Script Import Libraries
#---------------------------------------
import clr
clr.AddReference("IronPython.Modules.dll")
clr.AddReference("IronPython.SQLite.dll")

import os
import codecs
import json
import re
import sqlite3

#---------------------------------------
# Script Information
#---------------------------------------
ScriptName = "Twitch Subscription Tracker"
Website = "http://www.twitch.tv/ocgineer"
Description = "Tracks total amount of months, tiers and gifted Twitch subscriptions."
Creator = "Ocgineer"
Version = "1.0.3.0"

#---------------------------------------
# Script Global Variables
#---------------------------------------
SettingsFile = os.path.join(os.path.dirname(__file__), "settings.json")
ReadMeFile = os.path.join(os.path.dirname(__file__), "readme.txt")
DatabaseFile = os.path.join(os.path.dirname(__file__), "TSTDB.sqlite")

reUserName = re.compile(r"^[a-z0-9][a-z0-9_]{3,24}$")
reUserNotice = re.compile(r"(?:^(?:@(?P<irctags>[^\ ]*)\ )?:tmi\.twitch\.tv\ USERNOTICE)")
reCheckArgument = re.compile(r"^(?:(?P<t1>\d{1,3})\|(?P<t2>\d{1,3})\|(?P<t3>\d{1,3}))$")

#---------------------------------------
# Script Classes
#---------------------------------------
class Settings(object):
	""" Load in saved settings file if available else set default values. """
	def __init__(self, settingsfile=None):
		try:
			with codecs.open(settingsfile, encoding="utf-8-sig", mode="r") as f:
				self.__dict__ = json.load(f, encoding="utf-8")
		except:
			self.Permission = "Editor"
			self.SubbedTieredResponse = "{3} ({0},{1},{2})"
			self.GiftedTieredResponse = "{3} ({0},{1},{2})"
			self.TopDisplayAmount = 10

	def Reload(self, jsondata):
		""" Reload settings from the interface by given json data. """
		self.__dict__ = json.loads(jsondata, encoding="utf-8")
		return

class InstancedDatabase(object):
	""" Instanced database handler class. """
	def __init__(self, databasefile):
		self._connection = sqlite3.connect(databasefile, check_same_thread=False)
		self._cursor = self._connection.cursor()

	def execute(self, sqlquery, queryargs=None):
		""" Execute a sql query on the instanced database. """
		if queryargs:
			self._cursor.execute(sqlquery, queryargs)
		else:
			self._cursor.execute(sqlquery)
		return self._cursor

	def commit(self):
		""" Commit any changes of the instanced database. """
		self._connection.commit()

	def close(self):
		""" Close the instanced database connection. """
		self._connection.close()

	def __del__(self):
		""" Close the instanced database connection on destroy. """
		self._connection.close()

#---------------------------------------
# Chatbot Initialize Function
#---------------------------------------
def Init():

	# Globals
	global ScriptSettings
	global ScriptDB

	# Initialize settings
	ScriptSettings = Settings(SettingsFile)

	# Initialize intanced database
	ScriptDB = InstancedDatabase(DatabaseFile)

	# Create db file and/or table if not exists
	ScriptDB.execute("CREATE TABLE IF NOT EXISTS `TSTDATA` ("
										"`UserId` TEXT UNIQUE,"
										"`Username` TEXT,"
										"`Total` INTEGER,"
										"`1000` INTEGER,"
										"`2000` INTEGER,"
										"`3000` INTEGER,"
										"`ReceivedGifts` INTEGER,"
										"`GiftedTotal` INTEGER,"
										"`Gifted1000` INTEGER,"
										"`Gifted2000` INTEGER,"
										"`Gifted3000` INTEGER,"
										"PRIMARY KEY(UserId))")
	ScriptDB.commit()

	# End of Init
	return

#---------------------------------------
# Chatbot Save Settings Function
#---------------------------------------
def ReloadSettings(jsondata):

	# Reload newly saved settings
	ScriptSettings.Reload(jsondata)

	# End of ReloadSettings
	return

#---------------------------------------
# Chatbot Unload Function
#---------------------------------------
def Unload():

	# Close database
	ScriptDB.close()

	# End of Unload
	return

#---------------------------------------
# Chatbot Execute Function
#---------------------------------------
def Execute(data):

	# We only want raw data from Twitch to check for sub
	if data.IsRawData() and data.IsFromTwitch():

		# Apply regex on raw data to detect subscription usernotice
		usernotice = reUserNotice.search(data.RawData)
		if usernotice:

			# Parse IRCv3 tags in a dictionary
			tags = dict(re.findall(r"([^=]+)=([^;]*)(?:;|$)", usernotice.group("irctags")))
			# user-id														> User id of the subscriber/gifter
			# login															> User name of the subscriber/gifter
			# display-name											> Display name of the subscriber/gifter
			# msg-id														> Type of notice; sub, resub, charity, subgift
			# msg-param-months									> Amount of consecutive months
			# msg-param-sub-plan								> sub plan; prime, 1000, 2000, 3000
			# msg-param-recipient-id 						> user id of the gift receiver
			# msg-param-recipient-user-name			> user name of the gift receiver
			# msg-param-recipient-display-name	> display name of the gift receiver

			# Gifted subscription
			if tags["msg-id"] == "subgift":

				# Select the gifter from the database
				row = ScriptDB.execute("SELECT * FROM `TSTDATA` WHERE `UserId` = ?", (tags["user-id"],)).fetchone()

				# Gifter exists in DB
				if row:

					# Tier 3000 aka $25
					if tags["msg-param-sub-plan"] == "3000":
						GifterUpdate = (tags["login"], row[7] + 1, row[10] + 1, tags["user-id"])
						res = ScriptDB.execute("UPDATE `TSTDATA` SET `Username` = ?, `GiftedTotal` = ?, `Gifted3000` = ? WHERE `UserId` = ?", GifterUpdate)
					# Tier 2000 aka $10
					elif tags["msg-param-sub-plan"] == "2000":
						GifterUpdate	= (tags["login"], row[7] + 1, row[9] + 1, tags["user-id"])
						res = ScriptDB.execute("UPDATE `TSTDATA` SET `Username` = ?, `GiftedTotal` = ?, `Gifted2000` = ? WHERE `UserId` = ?", GifterUpdate)
					# Tier 1000 aka $5
					else:
						GifterUpdate	= (tags["login"], row[7] + 1, row[8] + 1, tags["user-id"])
						res = ScriptDB.execute("UPDATE `TSTDATA` SET `Username` = ?, `GiftedTotal` = ?, `Gifted1000` = ? WHERE `UserId` = ?", GifterUpdate)

					# Commit and log
					if res.rowcount > 0:
						ScriptDB.commit()
						Parent.Log(ScriptName, "Successfully updated {} in the database for gifting a sub.".format(tags["login"]))
					else:
						Parent.Log(ScriptName, "Failed to update {} in the database for gifting a sub.".format(tags["login"]))

				# Gifter does not exists in DB
				else:

					# Tier 3000 aka $25
					if tags["msg-param-sub-plan"] == "3000":
						GifterNewEntry = (tags["user-id"], tags["login"], 0, 0, 0, 0, 0, 1, 0, 0, 1)
					# Tier 2000 aka $10
					elif tags["msg-param-sub-plan"] == "2000":
						GifterNewEntry = (tags["user-id"], tags["login"], 0, 0, 0, 0, 0, 1, 0, 1, 0)
					# Tier 1000 aka $5
					else:
						GifterNewEntry = (tags["user-id"], tags["login"], 0, 0, 0, 0, 0, 1, 1, 0, 0)

					# Add new entry, commit and log
					if ScriptDB.execute("INSERT INTO `TSTDATA` VALUES (?,?,?,?,?,?,?,?,?,?,?)", GifterNewEntry).rowcount > 0:
						ScriptDB.commit()
						Parent.Log(ScriptName, "Succesfully added {} to the database for gifting a sub.".format(tags["login"]))
					# Insert failed
					else:
						Parent.Log(ScriptName, "Failed to add {} into the database for gifting a sub.".format(tags["login"]))

				# Select the receiver from the database
				row = ScriptDB.execute("SELECT * FROM `TSTDATA` WHERE `UserId` = ?", (tags["msg-param-recipient-id"],)).fetchone()

				# Receiver exists in DB
				if row:

					# Tier 3000 aka $25
					if tags["msg-param-sub-plan"] == "3000":
						ReceiverUpdate = (tags["msg-param-recipient-user-name"], row[2] + 1, row[5] + 1, row[6] + 1, tags["msg-param-recipient-id"])
						res = ScriptDB.execute("UPDATE `TSTDATA` SET `Username` = ?, `Total` = ?, `3000` = ?, `ReceivedGifts` = ? WHERE `UserId` = ?", ReceiverUpdate)
					# Tier 2000 aka $10
					elif tags["msg-param-sub-plan"] == "2000":
						ReceiverUpdate = (tags["msg-param-recipient-user-name"], row[2] + 1, row[4] + 1, row[6] + 1, tags["msg-param-recipient-id"])
						res = ScriptDB.execute("UPDATE `TSTDATA` SET `Username` = ?, `Total` = ?, `2000` = ?, `ReceivedGifts` = ? WHERE `UserId` = ?", ReceiverUpdate)
					# Tier 1000 aka $5
					else:
						ReceiverUpdate = (tags["msg-param-recipient-user-name"], row[2] + 1, row[3] + 1, row[6] + 1, tags["msg-param-recipient-id"])
						res = ScriptDB.execute("UPDATE `TSTDATA` SET `Username` = ?, `Total` = ?, `1000` = ?, `ReceivedGifts` = ? WHERE `UserId` = ?", ReceiverUpdate)

					# Commit and log
					if res.rowcount > 0:
						ScriptDB.commit()
						Parent.Log(ScriptName, "Successfully updated {} in the database for receiving a gifted sub.".format(tags["msg-param-recipient-user-name"]))
					else:
						Parent.Log(ScriptName, "Failed to update {} in the database for receiving a gifted sub.".format(tags["msg-param-recipient-user-name"]))

				# Receiver does not exists in DB
				else:

					# Tier 3000 aka $25
					if tags["msg-param-sub-plan"] == "3000":
						ReceiverNewEntry = (tags["msg-param-recipient-id"], tags["msg-param-recipient-user-name"], 1, 0, 0, 1, 1, 0, 0, 0, 0)
					# Tier 2000 aka $10
					elif tags["msg-param-sub-plan"] == "2000":
						ReceiverNewEntry = (tags["msg-param-recipient-id"], tags["msg-param-recipient-user-name"], 1, 0, 1, 0, 1, 0, 0, 0, 0)
					# Tier 1000 aka $5
					else:
						ReceiverNewEntry = (tags["msg-param-recipient-id"], tags["msg-param-recipient-user-name"], 1, 1, 0, 0, 1, 0, 0, 0, 0)

					# Add new entry, commit and log
					if ScriptDB.execute("INSERT INTO `TSTDATA` VALUES (?,?,?,?,?,?,?,?,?,?,?)", ReceiverNewEntry).rowcount > 0:
						ScriptDB.commit()
						Parent.Log(ScriptName, "Succesfully added {} to the database for receiving a gifted sub.".format(tags["msg-param-recipient-user-name"]))
					else:
						Parent.Log(ScriptName, "Failed to add {} into the database for receiving a gifted sub.".format(tags["msg-param-recipient-user-name"]))

			# Normal subscription
			else:

				# Extract amount of months if resub, set to 1 on new sub or error
				if tags["msg-id"] == "resub":
					try:
						cmonths = int(tags["msg-param-months"])
					except ValueError:
						cmonths = 1
				else:
					cmonths = 1

				# Select the subscriber from the database
				row = ScriptDB.execute("SELECT * FROM `TSTDATA` WHERE `UserId` = ?", (tags["user-id"],)).fetchone()

				# Subscriber exists in DB
				if row:

					# Auto correction total months of db contains less total months
					# than the consecutive months from Twitch (in case manual add)
					if row[2] < cmonths:
						tmonths = cmonths
						t1000 = cmonths - row[4] - row[5] - 1
					else:
						tmonths = row[2] + 1
						t1000 = row[3]

					# Tier 3000 aka 25$
					if tags["msg-param-sub-plan"] == "3000":
						updateEntry = (tags["login"], tmonths, t1000, row[5] + 1, tags["user-id"])
						res = ScriptDB.execute("UPDATE `TSTDATA` SET `Username` = ?, `Total` = ?, `1000` = ?, `3000` = ? WHERE `UserId` = ?", updateEntry)
					# Tier 2000 aka 10$
					elif tags["msg-param-sub-plan"] == "2000":
						updateEntry = (tags["login"], tmonths, t1000, row[4] + 1, tags["user-id"])
						res = ScriptDB.execute("UPDATE `TSTDATA` SET `Username` = ?, `Total` = ?, `1000` = ?, `2000` = ? WHERE `UserId` = ?", updateEntry)
					# Tier 1000 aka 5$ or Prime
					else:
						updateEntry = (tags["login"], tmonths, t1000 + 1, tags["user-id"])
						res = ScriptDB.execute("UPDATE `TSTDATA` SET `Username` = ?, `Total` = ?, `1000` = ? WHERE `UserId` = ?", updateEntry)

					# Commit and log
					if res.rowcount > 0:
						ScriptDB.commit()
						Parent.Log(ScriptName, "Succesfully updated {} in the database for subscribing.".format(tags["login"]))
					else:
						Parent.Log(ScriptName, "Failed to update {} in the database for subscribing.".format(tags["login"]))

				# Subscriber does not exists in DB
				else:

					# Tier 3000 aka 25$
					if tags["msg-param-sub-plan"] == "3000":
						if cmonths > 1:
							# add one month to tier 3000, rest to 1000
							newEntry = (tags["user-id"], tags["login"], cmonths, cmonths - 1, 0, 1, 0, 0, 0, 0, 0)
						else:
							newEntry = (tags["user-id"], tags["login"], cmonths, 0, 0, 1, 0, 0, 0, 0, 0)
					# Tier 2000 aka 10$
					elif tags["msg-param-sub-plan"] == "2000":
						if cmonths > 1:
							# add one month to tier 2000, rest to 1000
							newEntry = (tags["user-id"], tags["login"], cmonths, cmonths - 1, 1, 0, 0, 0, 0, 0, 0)
						else:
							newEntry = (tags["user-id"], tags["login"], cmonths, 0, 1, 0, 0, 0, 0, 0, 0)
					# Tier 1000 aka 5$ OR Prime
					else:
						newEntry = (tags["user-id"], tags["login"], cmonths, cmonths, 0, 0, 0, 0, 0, 0, 0)

					# Add new entry, commit and log
					if ScriptDB.execute("INSERT INTO `TSTDATA` VALUES (?,?,?,?,?,?,?,?,?,?,?)", newEntry).rowcount > 0:
						ScriptDB.commit()
						Parent.Log(ScriptName, "Succesfully added {} into the database for subscribing.".format(tags["login"]))
					# Insert failed
					else:
						Parent.Log(ScriptName, "Failed to add {} into the database for subscribing.".format(tags["login"]))

	# Command is given for editing database
	elif data.IsChatMessage() and data.GetParam(0).lower() == "!tst" and Parent.HasPermission(data.User, ScriptSettings.Permission.lower(), ""):

		# Command edit; edit database entry (by username)
		if data.GetParam(1).lower() == "editsubs" or data.GetParam(1).lower() == "editgifts":

			# Continue if given username is correct
			username = data.GetParam(2).strip('@').lower()
			if reUserName.search(username):

				# Verify given months argument '<tier1>|<tier2>|<tier3>'
				months = reCheckArgument.search(data.GetParam(3))
				if months:

					# Convert months to int and calc total
					tier1 = int(months.group("t1"))
					tier2 = int(months.group("t2"))
					tier3 = int(months.group("t3"))
					total = tier1 + tier2 + tier3

					# Select the username from the database
					row = ScriptDB.execute("SELECT * FROM `TSTDATA` WHERE `Username` = ?", (username,)).fetchone()

					# username exists in DB
					if row:

						# Create update tuple
						updateEntry = (total, tier1, tier2, tier3, username)

						# Execute update queuery for either editsub or editgift
						if data.GetParam(1).lower() == "editsubs":
							res = ScriptDB.execute("UPDATE `TSTDATA` SET `Total` = ?, `1000` = ?, `2000` = ?, `3000` = ? WHERE `Username` = ?", updateEntry)
						elif data.GetParam(1).lower() == "editgifts":
							res = ScriptDB.execute("UPDATE `TSTDATA` SET `GiftedTotal` = ?, `Gifted1000` = ?, `Gifted2000` = ?, `Gifted3000` = ? WHERE `Username` = ?", updateEntry)
						else:
							# should not come here...
							return

						# Commit and output status message
						if res.rowcount > 0:
							ScriptDB.commit()
							Parent.SendStreamMessage("Successfully updated {} in the database.".format(username))
						else:
							Parent.SendStreamMessage("Failed to update {} in the database, something went wrong!".format(username))

					# username does not exists in DB
					else:

						# Verify username exists on Twitch
						response = json.loads(Parent.GetRequest("https://decapi.me/twitch/id/{0}".format(username), {}))
						if response["status"] == 200:

							# Got user-id, username exists
							if response["response"].isdigit():

								# Create new entry tuple
								if data.GetParam(1).lower() == "editsubs":
									newEntry = (response["response"], username, total, tier1, tier2, tier3, 0, 0, 0, 0, 0)
								elif data.GetParam(1).lower() == "editgifts":
									newEntry = (response["response"], username, 0, 0, 0, 0, 0, total, tier1, tier2, tier3)
								else:
									# should not come here...
									return

								# Insert into database, commit and output status message
								if ScriptDB.execute("INSERT INTO `TSTDATA` VALUES (?,?,?,?,?,?,?,?,?,?,?)", newEntry).rowcount > 0:
									ScriptDB.commit()
									Parent.SendStreamMessage("Successfully added {} into the database.".format(username))
								else:
									Parent.SendStreamMessage("Failed to add {} into the database, something went wrong!".format(username))

							# Username does not exists
							else:
								Parent.SendStreamMessage("Given username does not exists on Twitch. Please use an existing username.")

						# API error
						else:
							Parent.SendStreamMessage("Failed to check Decapi API to verify username. Please try again later.")

				# Invalid months argument given
				else:
					Parent.SendStreamMessage("Invalid months given use; <tier1>|<tier2>|<tier3> where <tier> is a number in months of the respective tier e.g. 5|1|0")

			# Invalid username given
			else:
				Parent.SendStreamMessage("Invalid username given. Twitch usernames can only contain lower case letters, numbers and underscores and cannot start with an underscore.")

		# Command receivedsubs; edit the amount of received subs a person has
		elif data.GetParam(1).lower() == "receivedsubs":

			# Continue if given username is correct
			username = data.GetParam(2).strip('@').lower()
			if reUserName.search(username):

				# Verify amount is digit
				if data.GetParam(3).isdigit():

					# Select the username from the database
					row = ScriptDB.execute("SELECT * FROM `TSTDATA` WHERE `Username` = ?", (username,)).fetchone()

					# Username exists in DB
					if row:

						# amount
						try:
							amount = int(data.GetParam(3))
						except ValueError:
							return

						# Amount is lower or equal to total months in database
						if amount <= row[2]:

							# Execute update queuery for either editsub or editgift
							res = ScriptDB.execute("UPDATE `TSTDATA` SET `ReceivedGifts` = ? WHERE `Username` = ?", (amount, username))

							# Commit and output status message
							if res.rowcount > 0:
								ScriptDB.commit()
								Parent.SendStreamMessage("Successfully updated {} in the database.".format(username))
							else:
								Parent.SendStreamMessage("Failed to update {} in the database, something went wrong!".format(username))

						# Amount is higher than the total months in database
						else:
							Parent.SendStreamMessage("Invalid amount given. The amount is higher than the total subscribed months the user has in the database.")

					# Username does not exist in DB
					else:
						Parent.SendStreamMessage("Given username does not exist in database. Please add the user first by using !tst [editsubs|editgifts] <tier1|tier2|tier3>.")

				# Invalid amount given
				else:
					Parent.SendStreamMessage("Given amount is not a number, please give a valid number.")

			# Invalid username given
			else:
				Parent.SendStreamMessage("Invalid username given. Twitch usernames can only contain lower case letters, numbers and underscores and cannot start with an underscore.")

	# End of execute
	return

#---------------------------------------
# Chatbot Parameter Parser Function
#---------------------------------------
def Parse(parseString, user, target, message):

	# $tstSubbedTotal : Total subscribed months
	if "$tstSubbedTotal" in parseString:
		row = ScriptDB.execute("SELECT `Total` FROM `TSTDATA` WHERE `Username` = ?", (user,)).fetchone()
		return parseString.replace("$tstSubbedTotal", str(row[0]) if row else "0")

	# $tstSubbedT1 : Total subscribed tier 1 months
	if "$tstSubbedT1" in parseString:
		row = ScriptDB.execute("SELECT `1000` FROM `TSTDATA` WHERE `Username` = ?", (user,)).fetchone()
		return parseString.replace("$tstSubbedT1", str(row[0]) if row else "0")

	# $tstSubbedT2 : Total subscribed tier 2 months
	if "$tstSubbedT2" in parseString:
		row = ScriptDB.execute("SELECT `2000` FROM `TSTDATA` WHERE `Username` = ?", (user,)).fetchone()
		return parseString.replace("$tstSubbedT2", str(row[0]) if row else "0")

	# $tstSubbedT3 : Total subscribed tier 3 months
	if "$tstSubbedT3" in parseString:
		row = ScriptDB.execute("SELECT `3000` FROM `TSTDATA` WHERE `Username` = ?", (user,)).fetchone()
		return parseString.replace("$tstSubbedT3", str(row[0]) if row else "0")

	# $tstSubbedTiers : All tiers subscribed for
	if "$tstSubbedTiers" in parseString:
		row = ScriptDB.execute("SELECT * FROM `TSTDATA` WHERE `Username` = ?", (user,)).fetchone()
		if row:
			return parseString.replace("$tstSubbedTiers", ScriptSettings.SubbedTieredResponse.format(row[3], row[4], row[5], row[2], row[6]))
		else:
			return parseString.replace("$tstSubbedTiers", ScriptSettings.SubbedTieredResponse.format("0", "0", "0", "0", "0"))

	# $tstReceivedTotal : Total received gifted subscriptions
	if "$tstReceivedTotal" in parseString:
		row = ScriptDB.execute("SELECT `ReceivedGifts` FROM `TSTDATA` WHERE `Username` = ?", (user,)).fetchone()
		return parseString.replace("$tstReceivedTotal", str(row[0]) if row else "0")

	# $tstGiftedTotal : Total gifted gifted subscriptions
	if "$tstGiftedTotal" in parseString:
		row = ScriptDB.execute("SELECT `GiftedTotal` FROM `TSTDATA` WHERE `Username` = ?", (user,)).fetchone()
		return parseString.replace("$tstGiftedTotal", str(row[0]) if row else "0")

	# $tstGiftedT1 : Total gifted tier 1 subscriptions
	if "$tstGiftedT1" in parseString:
		row = ScriptDB.execute("SELECT `Gifted1000` FROM `TSTDATA` WHERE `Username` = ?", (user,)).fetchone()
		return parseString.replace("$tstGiftedT1", str(row[0]) if row else "0")

	# $tstGiftedT2 : Total gifted tier 2 subscriptions
	if "$tstGiftedT2" in parseString:
		row = ScriptDB.execute("SELECT `Gifted2000` FROM `TSTDATA` WHERE `Username` = ?", (user,)).fetchone()
		return parseString.replace("$tstGiftedT2", str(row[0]) if row else "0")

	# $tstGiftedT3 : Total gifted tier 3 subscriptions
	if "$tstGiftedT3" in parseString:
		row = ScriptDB.execute("SELECT `Gifted3000` FROM `TSTDATA` WHERE `Username` = ?", (user,)).fetchone()
		return parseString.replace("$tstGiftedT3", str(row[0]) if row else "0")

	# $tstGiftedTiers : All gifted tiers
	if "$tstGiftedTiers" in parseString:
		row = ScriptDB.execute("SELECT * FROM `TSTDATA` WHERE `Username` = ?", (user,)).fetchone()
		if row:
			return parseString.replace("$tstGiftedTiers", ScriptSettings.GiftedTieredResponse.format(row[8], row[9], row[10], row[7]))
		else:
			return parseString.replace("$tstGiftedTiers", ScriptSettings.GiftedTieredResponse.format("0", "0", "0", "0"))

	# $tstTopSubbers
	if "$tstTopSubbers" in parseString:
		rows = ScriptDB.execute("SELECT `Username`, `Total` FROM `TSTDATA` WHERE `Total` > 0 ORDER BY `Total` DESC, `Username` LIMIT 0, ?", (ScriptSettings.TopDisplayAmount,)).fetchall()
		if rows:
			tlist = []
			for item in rows:
				tlist.append("{} ({})".format(item[0], item[1]))
			return parseString.replace("$tstTopSubbers", ", ".join(tlist)[:514 - len(parseString)])
		else:
			return parseString.replace("$tstTopSubbers", "no data available")

	# $tstTopGifters
	if "$tstTopGifters" in parseString:
		rows = ScriptDB.execute("SELECT `Username`, `GiftedTotal` FROM `TSTDATA` WHERE `GiftedTotal` > 0 ORDER BY `GiftedTotal` DESC, `Username` LIMIT 0, ?", (ScriptSettings.TopDisplayAmount,)).fetchall()
		if rows:
			tlist = []
			for item in rows:
				tlist.append("{} ({})".format(item[0], item[1]))
			return parseString.replace("$tstTopGifters", ", ".join(tlist)[:514 - len(parseString)])
		else:
			return parseString.replace("$tstTopGifters", "no data available")

	# Return unaltered parseString
	return parseString

#---------------------------------------
# Chatbot Tick Function
#---------------------------------------
def Tick():
	return

#---------------------------------------
# Script UI Button Functions
#---------------------------------------
def OpenReadMe():
	os.startfile(ReadMeFile)
	return
